/*global define */
/*jslint white: true */

/*
	math utils:

	This file implements math related utilities.
*/

define(["lodash"],
function (lodash) {
	"use strict";

	var radiansPerDegree = Math.PI / 180,
		degreesPerRadian = 180 / Math.PI;


	function deg(inRadians) {
		return degreesPerRadian * inRadians;
	}

	function rad(inDegrees) {
		return radiansPerDegree * inDegrees;
	}

	function lerp(a, b, t) {
		if (lodash.isArray(a) && lodash.isArray(b)) {
			var i, result = [];
			for (i = 0; i < a.length; i += 1) {
				result[i] = a[i] + t * (b[i] - a[i]);
			}
			return result;
		}
		return a + t * (b - a);
	}

	// t - between 0 and 1 (0 returns x1, 1 returns x2)
	// tension - between 1 and -1.
	function cubicLerp (x0, x1, x2, x3, t, tension) {
		var t0,t1,t2,t3,a0,a1,a2,a3, normalizedTension = (1.0-tension) * 0.5;

		t2 = t * t;
		t3 = t2 * t;
		t0 = (x1-x0) * normalizedTension;
		t0 += (x2-x1) * normalizedTension;
		t1 = (x2-x1) * normalizedTension;
		t1 += (x3-x2) * normalizedTension;
		a0 = 2.0*t3 - 3.0*t2 + 1.0;
		a1 = t3 - 2.0*t2 + t;
		a2 = t3 - t2;
		a3 = -2.0*t3 + 3.0*t2;

		return a0*x1 + a1*t0 + a2*t1 + a3*x2;
	}

	function add(a, b) {
		if (lodash.isArray(a) && lodash.isArray(b)) {
			var i, result = [];
			for (i = 0; i < a.length; i += 1) {
				result[i] = a[i] + b[i];
			}
			return result;
		}
		return a+b;
	}

	function multiplyByScalar(a, b) {
		if (lodash.isArray(a)) {
			var i, result = [];
			for (i = 0; i < a.length; i += 1) {
				result[i] = a[i] * b;
			}
			return result;
		}
		return a*b;
	}

	// Based on porter duff, this is the alpha calculation.
	function getCompositeWeights(a, b) {
		var	aWeight = a, bWeight = b * (1 - a);
		return [aWeight, bWeight];
	}

	function getCompositeWeightSum(a, b) {
		var	aWeight = a, bWeight = b * (1 - a);
		return aWeight + bWeight;
	}

	// computes y = x * x * (3.0 - 2.0 * x) to ease in/out between 0-1
	function easeInAndOut(t) {
		if (t < 0) return 0.0;
		if (t > 1) return 1.0;
		
		return t * t * (3.0 - 2.0 * t);
	}

	function equalWithinEpsilon(inX1F, inX2F, inEpsilonF) {
		inEpsilonF = inEpsilonF || 1e-10;

		return Math.abs(inX1F - inX2F) < inEpsilonF;
	}

	function equalToZeroWithinEpsilon(inValueF, inEpsilonF) {
		return equalWithinEpsilon(inValueF, 0.0, inEpsilonF);
	}

	function cubeRoot(inValueF) {
		return Math.pow(inValueF, 1 / 3);
	}

	function solveLinearRoot(a, b) {
		var root = 0;
		if (equalToZeroWithinEpsilon(a)) {
			root = 0.0;				/* irrational */
		} else {
			root = -b / a;			/* one root */
		}
		return root;
	}

	function solveOneQuadraticRoot(a, b, c) {
		var root, atimes2 = 2.0 * a, ctimes2 = 2.0 * c, e = b * b - atimes2 * ctimes2, roots = [0, 0];

		if (equalToZeroWithinEpsilon(a)) {
			root = solveLinearRoot(b, c, root);
		} else {
			if (e < 0.0) {
				throw new Error("No rational roots.");
			} else {
				if (equalToZeroWithinEpsilon(e)) {			/* one root           */
					root = -b / atimes2;
				} else {									/* two roots          */
					e = Math.sqrt(e);
					if (b > 0.0) {							/* make answer stable */
						roots[0] = -ctimes2 / (e + b);
						roots[1] = -(e + b) / atimes2;
					} else {								/* b <= 0             */
						roots[0] = ctimes2 / (e - b);
						roots[1] = (e - b) / atimes2;
					}
					if (roots[0] >= 0.0 && roots[0] <= 1.0) {
						root = roots[0];
					} else if (roots[1] >= 0.0 && roots[1] <= 1.0) {
						root = roots[1];
					}
				}
			}
		}
		return root;
	}

	function solveOneCubicRoot(a, b, c, d) {
		var root = 0, a1, a2, a3, q, q_cube, r, r_sqr, a1_third, phi, tmp, roots = [0, 0, 0];

		if (equalToZeroWithinEpsilon(a)) {
			root = solveOneQuadraticRoot(b, c, d);
		} else {
			a1 = b / a;
			a2 = c / a;
			a3 = d / a;
			a1_third = a1 / 3.0;
			q = (a1 * a1 - 3.0 * a2) / 9.0;
			q_cube = q * q * q;
			r = (2.0 * a1 * a1 * a1 - 9.0 * a1 * a2 + 27.0 * a3) / 54.0;
			r_sqr = r * r;

			if (q_cube >= r_sqr) {						/* two or three roots */
				if (equalToZeroWithinEpsilon(q)) {		/* degenerate case of */
					root = -a1_third;					/* one triple root    */
				} else {
					phi = Math.acos(r / Math.sqrt(q_cube));
					tmp = -2.0 * Math.sqrt(q);
					roots[0] = tmp * Math.cos(phi / 3.0) - a1_third;
					roots[1] = tmp * Math.cos((phi + Math.PI * 2) / 3.0) - a1_third;
					roots[2] = tmp * Math.cos((phi + 4.0 * Math.PI) / 3.0) - a1_third;

					if (roots[0] > 0.0 && roots[0] < 1.0) {
						root = roots[0];
					} else if (roots[1] >= 0.0 && roots[1] <= 1.0) {
						root = roots[1];
					} else if (roots[2] >= 0.0 && roots[2] <= 1.0) {
						root = roots[2];
					}
				}
			} else if (r >= 0.0) {
				tmp = cubeRoot(Math.sqrt(r_sqr - q_cube) + r);
				root = -tmp - q / tmp - a1_third;
			} else {
				tmp = cubeRoot(Math.sqrt(r_sqr - q_cube) - r);
				root = tmp + q / tmp - a1_third;
			}
		}
		return root;
	}

	function loop(inT, inLoopType) {
		if (inLoopType !== undefined) {
			if (inT < 0) {
				inT = Math.abs(inT);
				inT = Math.floor(inT) + (1 - (inT % 1));
			}
			if (inLoopType === "cycle") {
				inT = inT % 1;
			} else if (inLoopType === "pingpong") {
				inT = inT % 2;
				if (inT > 1) {
					inT = 2 - inT;
				}
			}
		} else {
			inT = Math.max(0, inT);
			inT = Math.min(1, inT);
		}
		return inT;
	}

	function intersectLines(pt1, pt2, pt3, pt4) {
		var result = [], den, cross12, cross34;

		den = (pt1[0] - pt2[0]) * (pt3[1] - pt4[1]) - (pt1[1] - pt2[1]) * (pt3[0] - pt4[0]);

		if (den) {
			cross12 = (pt1[0] * pt2[1] - pt1[1] * pt2[0]);
			cross34 = (pt3[0] * pt4[1] - pt3[1] * pt4[0]);
			result[0] = (cross12 * (pt3[0] - pt4[0]) - cross34 * (pt1[0] - pt2[0])) / den;
			result[1] = (cross12 * (pt3[1] - pt4[1]) - cross34 * (pt1[1] - pt2[1])) / den;
			return result;
		}

		// Line are parallel, return unidentified.
		return undefined;
	}

	function log10(n) {
		return Math.log(n) / Math.log(10);
	}

	function normalizedToDecibel(inNormalized) {
		// Normalized zero is equivalent to decibel -infinity, so we limit to a very small value which approaches zero.
		if (inNormalized < 1e-15) {
			inNormalized = 1e-15; // (-300dB)
		}

		// Normalized one is equivalent to decibel zero.
		return 20.0 * log10(inNormalized);
	}

	function decibelToNormalized(inDecibel) {
		// The smaller the negative decibel value, the closer we'll get to normalized zero.
		// However, this routine will never actually return true normalized zero.
		return Math.pow(10.0, inDecibel / 20.0);
	}

	function angle (rotation) {
		var result = Math.atan2(rotation[1], rotation[0]);
		return result;
	}

	return {
		oneThird: 1.0 / 3.0,
		lerp: lerp,
		cubicLerp: cubicLerp,
		add: add,
		multiplyByScalar: multiplyByScalar,
		getCompositeWeights: getCompositeWeights,
		getCompositeWeightSum: getCompositeWeightSum,
		easeInAndOut: easeInAndOut,
		loop: loop,
		equalWithinEpsilon: equalWithinEpsilon,
		equalToZeroWithinEpsilon: equalToZeroWithinEpsilon,
		cubeRoot: cubeRoot,
		solveLinearRoot: solveLinearRoot,
		solveOneQuadraticRoot: solveOneQuadraticRoot,
		solveOneCubicRoot: solveOneCubicRoot,
		intersectLines: intersectLines,
		deg: deg,
		rad: rad,
		log10: log10,
		normalizedToDecibel: normalizedToDecibel,
		decibelToNormalized: decibelToNormalized,
		angle : angle
	};
});
